/**
 * Broker JS API for TradingView Trading Platform
 * @packageDocumentation
 * @module Broker
 */
// Generated by dts-bundle-generator v7.0.0

export declare const enum CommonAccountManagerColumnId {
	/** You should use this column id if you want to fix the column in the positions and orders tables. */
	Symbol = "symbol"
}
export declare const enum ConnectionStatus {
	Connected = 1,
	Connecting = 2,
	Disconnected = 3,
	Error = 4
}
export declare const enum ConnectionType {
	Demo = 1,
	Real = 0
}
export declare const enum NotificationType {
	Error = 0,
	Success = 1
}
export declare const enum OrderStatus {
	Canceled = 1,
	Filled = 2,
	Inactive = 3,
	Placing = 4,
	Rejected = 5,
	Working = 6
}
export declare const enum OrderStatusFilter {
	All = 0,
	Canceled = 1,
	Filled = 2,
	Inactive = 3,
	Rejected = 5,
	Working = 6
}
export declare const enum OrderTicketFocusControl {
	LimitPrice = 1,
	StopPrice = 2,
	TakeProfit = 3,
	StopLoss = 4,
	Quantity = 5
}
export declare const enum OrderType {
	Limit = 1,
	Market = 2,
	Stop = 3,
	StopLimit = 4
}
export declare const enum ParentType {
	Order = 1,
	Position = 2,
	IndividualPosition = 3
}
export declare const enum PriceType {
	Limit = 1,
	Stop = 2
}
export declare const enum Side {
	Buy = 1,
	Sell = -1
}
/**
 * Enumeration of the built-in formatters for the Account Manager columns.
 * Refer to the [Value formatters](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/value-formatters.md) section for more information.
 */
export declare const enum StandardFormatterName {
	Date = "date",
	DateOrDateTime = "dateOrDateTime",
	Default = "default",
	Fixed = "fixed",
	FixedInCurrency = "fixedInCurrency",
	VariablePrecision = "variablePrecision",
	FormatQuantity = "formatQuantity",
	FormatPrice = "formatPrice",
	FormatPriceForexSup = "formatPriceForexSup",
	IntegerSeparated = "integerSeparated",
	LocalDate = "localDate",
	LocalDateOrDateTime = "localDateOrDateTime",
	Percentage = "percentage",
	Pips = "pips",
	Profit = "profit",
	ProfitInInstrumentCurrency = "profitInInstrumentCurrency",
	Side = "side",
	PositionSide = "positionSide",
	Status = "status",
	Symbol = "symbol",
	Text = "text",
	Type = "type",
	MarginPercent = "marginPercent",
	Empty = "empty"
}
export declare enum OrderOrPositionMessageType {
	Information = "information",
	Warning = "warning",
	Error = "error"
}
export declare enum StopType {
	StopLoss = 0,
	TrailingStop = 1
}
/**
 * This is the generic type useful for declaring a nominal type,
 * which does not structurally matches with the base type and
 * the other types declared over the same base type
 *
 * Usage:
 * @example
 * type Index = Nominal<number, 'Index'>;
 * // let i: Index = 42; // this fails to compile
 * let i: Index = 42 as Index; // OK
 * @example
 * type TagName = Nominal<string, 'TagName'>;
 */
export declare type Nominal<T, Name extends string> = T & { /* eslint-disable-next-line jsdoc/require-jsdoc */
	[Symbol.species]: Name;
};
/** Column properties for the [Account Manager](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md) pages. */
export interface AccountManagerColumnBase<TFormatterName extends StandardFormatterName | FormatterName> {
	/** Column title. It will be displayed in the table's header row. */
	label: string;
	/**
	 * Horizontal alignment of the cell value. The default value is `left`.
	 *
	 * | Alignment    |   Description  |
	 * |--------------|----------------|
	 * | left         | It aligns the cell value to the left. |
	 * | right        | It aligns the cell value to the right. |
	 */
	alignment?: CellAlignment;
	/** Unique column identifier. */
	id: string;
	/**
	 * Defines a formatter to be applied for data formatting, which can either belong to the `StandardFormatterName` or `FormatterName` type.
	 * If no specific formatter is set, the value is displayed as is.
	 *
	 * Default formatter names are enumerated in {@link StandardFormatterName}.
	 * Refer to the [Default formatters](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/value-formatters.md#default-formatters) section to see the full list of formatters.
	 *
	 * You can also create custom formatters using the {@link AccountManagerInfo.customFormatters} property.
	 */
	formatter?: TFormatterName;
	/**
	 * The `dataFields` array contains fields from an order/position data object.
	 * `dataFields` is used to generate the values displayed in a column.
	 * The displayed value in the column updates only when the corresponding values in the data object change.
	 *
	 * If no {@link formatter} is specified, the displayed values will be space-separated in the column.
	 * When a `formatter` is defined, it processes only the specified values.
	 * If an empty array is assigned to `dataFields`, the `formatter` will receive the entire data object.
	 *
	 * **Example**
	 *
	 * - For a column with `dataFields` set as `['avgPrice', 'qty']`, the displayed value updates only when the `avgPrice` or `qty` values in the data object change.
	 * - For a column with an empty `dataFields` array, the displayed value updates if any values in the data object change.
	 */
	dataFields: TFormatterName extends StandardFormatterName ? StandardFormattersDependenciesMapping[TFormatterName] : string[];
	/**
	 * Data object key that is used for data sorting.
	 *
	 * If `sortProp` is not provided, the first element of the `dataFields` array will be used.
	 * If the `dataFields` array is empty, the column sorting will be unavailable.
	 */
	sortProp?: string;
	/** When set to `true` will prevent column sorting. */
	notSortable?: boolean;
	/** Tooltip string for the column. */
	help?: string;
	/**
	 * `highlightDiff` can be set with [`StandardFormatterName.FormatPrice`](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/value-formatters.md#formatPrice)
	 * and [`StandardFormatterName.FormatPriceForexSup`](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/value-formatters.md#formatPriceForexSup) formatters to highlight the changes of the field.
	 * If `highlightDiff is `true`, the custom formatters will also get previous values.
	 */
	highlightDiff?: boolean;
	/** Setting `notHideable` to `true` prevents the column from being hidden. */
	notHideable?: boolean;
	/** Setting `hideByDefault` to `true` hides the column by default. */
	hideByDefault?: boolean;
	/** Key of the row object that is used to get the tooltip to display when hovering over a cell.
	 * The tooltip property refers to an object whose keys are property names and
	 * values are the corresponding tooltips.
	 */
	tooltipProperty?: string;
	/** If set to `true`, the first character of every word in the sentence in the column
	 * will be capitalized. The default value is `true`.
	 */
	isCapitalize?: boolean;
	/** Setting `showZeroValues` to `true` hides any zero values. The default value is `true`. */
	showZeroValues?: boolean;
}
/**
 * The information object that is used to build the [Account Manager](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md).
 */
export interface AccountManagerInfo {
	/** Name of the broker */
	accountTitle: string;
	/**
	 * Custom fields that are always displayed at the top-right corner of the Account Manager.
	 * Refer to the [Account Summary row](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md#account-summary-row) section for more information.
	 */
	summary: AccountManagerSummaryField[];
	/**
	 * An optional array for defining [custom formatters](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/value-formatters.md#custom-formatters).
	 * Each formatter description is an object with the following fields:
	 *
	 * - `name` ({@link FormatterName}): Unique formatter name.
	 *
	 * - `formatText` ({@link TableFormatTextFunction}):
	 * Function that is used for formatting a cell value to a string.
	 * The `formatText` field is required because it is used to generate exported data.
	 * You can return an empty string if you do not need this function.
	 *
	 * - `formatElement` ({@link CustomTableFormatElementFunction} | `undefined`):
	 * Optional function that is used for formatting a cell value to a string or an HTML element.
	 *
	 * If the `formatElement` function is provided, it only handles the formatting of displayed values.
	 * Otherwise the `formatText` function is used.
	 * For optimal performance, it is recommended to only use `formatText` if you intend to display only string values.
	 *
	 * **Example**
	 * ```ts
	 * {
	 *     name: 'closeButton' as FormatterName, // Typecast to FormatterName. Use constant in real code
	 *     formatText: () => '', // Returns an empty string because we don't need to display this in the exported data
	 *     formatElement: ({ values: [id] }: TableFormatterInputs<[id: string]>) => {
	 *         const button = document.createElement('button');
	 *
	 *         button.innerText = 'Close';
	 *
	 *         button.addEventListener('click', () => {
	 *             event.stopPropagation();
	 *
	 *             closePosition(id);
	 *         });
	 *
	 *         return button;
	 *     },
	 * }
	 * ```
	 */
	customFormatters?: CustomTableElementFormatter[];
	/** An array of data objects that create columns for the [Orders](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md#orders-and-positions) page.
	 * You can display any field of an {@link Order}
	 * or add your own fields to an order object and display them.
	 */
	orderColumns: OrderTableColumn[];
	/** Optional sorting of the orders table. */
	orderColumnsSorting?: SortingParameters;
	/**
	 * An array of data objects that create columns for the [History](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md#history) page where all orders from previous sessions are shown.
	 * Note that this page is only shown
	 * if you set the {@link BrokerConfigFlags.supportOrdersHistory} to `true`
	 * and implement the {@link IBrokerTerminal.ordersHistory} method.
	 */
	historyColumns?: AccountManagerColumn[];
	/** Optional sorting of the table on the [History](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md#history) page. */
	historyColumnsSorting?: SortingParameters;
	/**
	 * An array of data objects that create columns for the [Positions](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md#orders-and-positions) page.
	 * You can display any field of a {@link Position}
	 * or add your own fields to a position object and display them.
	 */
	positionColumns?: AccountManagerColumn[];
	/**
	 * You can display any field of an {@link IndividualPosition}
	 * or add your own fields to an individualPosition object and display them.
	 */
	individualPositionColumns?: AccountManagerColumn[];
	/** Adds [custom pages](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md#custom-pages) to the Account Manager. Each page is a set of tables. */
	pages: AccountManagerPage[];
	/** Optional list of statuses to be used in the orders filter. Default list is used if it hasn't been set. */
	possibleOrderStatuses?: OrderStatus[];
	/** Margin used */
	marginUsed?: IWatchedValue<number>;
	/**
	 * Optional function to create a custom context menu.
	 * @param contextMenuEvent - MouseEvent or TouchEvent object passed by a browser
	 * @param activePageActions - array of `ActionMetaInfo` items for the current page
	 * @returns `Promise` that is resolved with an array of `ActionMetaInfo`
	 */
	contextMenuActions?(contextMenuEvent: MouseEvent | TouchEvent, activePageActions: ActionMetaInfo[]): Promise<ActionMetaInfo[]>;
}
/** A description of an additional Account Manager tab. */
export interface AccountManagerPage {
	/** Unique identifier of a page */
	id: string;
	/** Page title. It is the tab name. */
	title: string;
	/** It is possible to display one or more tables in this tab. */
	tables: AccountManagerTable[];
}
/** Custom field that will always be shown above the pages of the Account manager */
export interface AccountManagerSummaryField {
	/** Text to display for the summary field */
	text: string;
	/** A WatchedValue object that can be used to read the state of field. */
	wValue: IWatchedValueReadonly<any>;
	/**
	 * Name of the formatter to be used for data formatting. If `formatter` is not
	 * set the value is displayed as is. Formatter can be a default or a custom one.
	 */
	formatter?: StandardFormatterName;
	/** Optional parameter which can be set to display the field by default. */
	isDefault?: boolean;
}
/**
 * Account Summary table meta-info
 * **NOTE**: make sure that you have a unique string `id` field in each row to identify it.
 */
export interface AccountManagerTable {
	/** Unique identifier of a table. */
	id: string;
	/** Optional title of a table. */
	title?: string;
	/** Table columns */
	columns: AccountManagerColumn[];
	/** Optional sorting of the table. If set, then the table will be sorted by these parameters, if the user has not enabled sorting by a specific column. */
	initialSorting?: SortingParameters;
	/** This delegate is used to watch the data changes and update the table.
	 * Pass new account manager data row by row to the `fire` method of the delegate.
	 */
	changeDelegate: ISubscription<(data: {}) => void>;
	/** Option flags for the table. */
	flags?: AccountManagerTableFlags;
	/**
	 * This function is used to request table data. It should return Promise (or Deferred) and resolve it with an array of data rows.
	 *
	 * Each row is an object. Keys of this object are column names with the corresponding values.
	 *
	 * There is a predefined field `isTotalRow` which can be used to mark a row that should be at the bottom of the table.
	 * @param paginationLastId - Last pagination id
	 */
	getData(paginationLastId?: string | number): Promise<{}[]>;
}
/** Boolean options for the account manager table */
export interface AccountManagerTableFlags {
	/** Does the table support pagination */
	supportPagination?: boolean;
}
export interface AccountMetainfo {
	/**
	 * Account ID
	 */
	id: AccountId;
	/**
	 * Account Name
	 */
	name: string;
	/**
	 * Currency
	 */
	currency?: string;
	/**
	 * Currency Sign (Prefix)
	 */
	currencySign?: string;
}
export interface ActionDescription {
	/** Displayed text for action */
	text?: "-" | string;
	/** Is a menu separator */
	separator?: boolean;
	/** Keyboard shortcut for the action. Displayed as hint text. */
	shortcut?: string;
	/** Tooltip text to be displayed when hovering over the action item. */
	tooltip?: string;
	/** Value of the checkbox. */
	checked?: boolean;
	/** Getter to retrieve the current checkbox value. */
	checkedStateSource?: () => boolean;
	/** Whether menu action represents a checkbox state. Set it to true if you need a checkbox. */
	checkable?: boolean;
	/** Whether the action is enabled. Set to false to disabled the action. */
	enabled?: boolean;
	/** External link (url) which will be opened upon clicking the menu item. */
	externalLink?: boolean;
	/**
	 * A string of SVG icon for an action. A string should be a string representation of SVG (not a path/URL).
	 */
	icon?: string;
}
/** Menu action which provides a callback function to be executed. Action is executed when user clicks the item. */
export interface ActionDescriptionWithCallback extends ActionDescription {
	/** Action to be executed when user clicks the menu item. */
	action: (a?: ActionDescription) => void;
}
export interface BaseInputFieldValidatorResult {
	/** Is the base input value valid */
	valid: boolean;
}
/**
 * An object that contains information about a bracket order.
 */
export interface BracketOrder extends BracketOrderBase, CustomFields {
}
export interface BracketOrderBase extends PlacedOrderBase {
	/** If an order is a bracket, it should contain an ID of a parent order/position. */
	parentId: string;
	/** Type of the bracket's parent. */
	parentType: ParentType;
}
export interface Brackets {
	/** Stop loss */
	stopLoss?: number;
	/** Take Profit */
	takeProfit?: number;
	/** Trailing Stop Pips */
	trailingStopPips?: number;
}
export interface BrokerConfigFlags {
	/**
	 * Changes _Amount_ to _Quantity_ in Order Ticket.
	 * @default false
	 */
	showQuantityInsteadOfAmount?: boolean;
	/**
	 * Enables order brackets: take-profit and stop-loss.
	 * @default false
	 */
	supportOrderBrackets?: boolean;
	/**
	 * Enables trailing stop orders.
	 * If you set this flag to `true`, the library displays trailing stop orders and a user can place a trailing stop order using the Order Ticket.
	 * @default false
	 */
	supportTrailingStop?: boolean;
	/**
	 * Enables positions.
	 * This flag requires the {@link IBrokerTerminal.positions} method to be implemented.
	 * If you set `supportPositions` to `false`, the _Positions_ tab in the [Account Manager](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md) will be hidden.
	 * @default true
	 */
	supportPositions?: boolean;
	/**
	 * Enables position brackets: take-profit and stop-loss orders.
	 * If you set `supportPositionBrackets` to `true`, the library displays an _Edit_ button for positions and _Edit position..._ in the position's context menu.
	 * This flag requires the {@link IBrokerTerminal.editPositionBrackets} method to be implemented.
	 * @default false
	 */
	supportPositionBrackets?: boolean;
	/**
	 * Enables brackets for individual positions: take-profit and stop-loss orders.
	 * If you set this flag to `true`, the library displays an _Edit_ button for individual positions and _Edit position..._ in the individual position's context menu.
	 * This flag requires the {@link IBrokerTerminal.editIndividualPositionBrackets} method to be implemented.
	 * @default false
	 */
	supportIndividualPositionBrackets?: boolean;
	/**
	 * Enables individual and net positions.
	 * If you set this flag to `true`, the [Account Manager](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md) will have two tabs: _Individual Positions_ and _Net Positions_.
	 * This flag requires the {@link IBrokerTerminal.individualPositions} method to be implemented.
	 *
	 * @default false
	 */
	supportPositionNetting?: boolean;
	/**
	 * Enables position closing.
	 * This flag requires the {@link IBrokerTerminal.closePosition} method to be implemented.
	 * If `supportClosePosition` is set to `true`, the library displays a close button and calls the `closePosition` method.
	 * If `supportClosePosition` is set to `false`, the library displays a close button but calls the {@link IBrokerTerminal.placeOrder} method with the `isClose` property set to `true`.
	 * @default false
	 */
	supportClosePosition?: boolean;
	/**
	 * Enables individual position closing.
	 * This flag requires the {@link IBrokerTerminal.closeIndividualPosition} method to be implemented.
	 * @default false
	 */
	supportCloseIndividualPosition?: boolean;
	/**
	 * Enables order price editing.
	 * If you set this flag to `false`, the price control in the _Order Ticket_ will be disabled when users modify orders.
	 * @default true
	 */
	supportModifyOrderPrice?: boolean;
	/**
	 * Enables order quantity editing.
	 * If you set this flag to `false`, the quantity control in the _Order Ticket_ will be disabled when users modify orders.
	 * @default true
	 */
	supportEditAmount?: boolean;
	/**
	 * Enables order brackets editing.
	 * If you set this flag to `false`, the bracket's control in the Order Ticket will be disabled,
	 * and the _Modify_ button will be hidden from the chart and in the [Account Manager](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md).
	 * @default true
	 */
	supportModifyBrackets?: boolean;
	/**
	 * Enables Level 2 data for the Depth of Market (DOM) widget.
	 * This flag requires the [`subscribeDepth`](https://www.tradingview.com/charting-library-docs/latest/connecting_data/Datafeed-API#subscribedepth) and [`unsubscribeDepth`](https://www.tradingview.com/charting-library-docs/latest/connecting_data/Datafeed-API#unsubscribedepth) methods to be implemented.
	 * Note that you should also enable the {@link BrokerConfigFlags.supportDOM} flag to display the DOM widget in the UI.
	 * @default false
	 */
	supportLevel2Data?: boolean;
	/**
	 * Enables the Depth of Market (DOM) widget.
	 * Note that you should also enable the {@link BrokerConfigFlags.supportLevel2Data} flag and implement the [`subscribeDepth`](https://www.tradingview.com/charting-library-docs/latest/connecting_data/Datafeed-API#subscribedepth) and [`unsubscribeDepth`](https://www.tradingview.com/charting-library-docs/latest/connecting_data/Datafeed-API#unsubscribedepth) methods to provide Level 2 data for the DOM widget.
	 * @default false
	 */
	supportDOM?: boolean;
	/**
	 * Enables multiple positions for one instrument at the same time.
	 * Supporting multiple positions prevents creating the default implementation for a reversing position.
	 * @default false
	 */
	supportMultiposition?: boolean;
	/**
	 * Allows you to use your own Profit & Loss (P&L) values for positions.
	 * If `supportPLUpdate` is set to `true`, you should call the {@link IBrokerConnectionAdapterHost.plUpdate} method as soon as P&L values are changed.
	 * If `supportPLUpdate` is set to `false`, the library automatically calculates P&L values as the difference between the current trade and the average position price.
	 * @default true
	 */
	supportPLUpdate?: boolean;
	/**
	 * Enables position reversing.
	 * If `supportReversePosition` is set to `false`, the _Reverse Position_ button will be hidden from the UI.
	 * @default false
	 */
	supportReversePosition?: boolean;
	/**
	 * Enables native position reversing.
	 * This flag requires the {@link IBrokerTerminal.reversePosition} method to be implemented.
	 * If `supportNativeReversePosition` is set to `false`, the library expects you to place a reversing order via the {@link IBrokerTerminal.placeOrder} method.
	 * @default false
	 */
	supportNativeReversePosition?: boolean;
	/**
	 * Enables market orders type in the Order Ticket.
	 * @default true
	 */
	supportMarketOrders?: boolean;
	/**
	 * Enables limit orders type in the Order Ticket.
	 * @default true
	 */
	supportLimitOrders?: boolean;
	/**
	 * Enables stop orders type in the Order Ticket.
	 * @default true
	 */
	supportStopOrders?: boolean;
	/**
	 * Enables stop-limit orders type in the Order Ticket.
	 * @default false
	 */
	supportStopLimitOrders?: boolean;
	/**
	 * Enables demo live switcher.
	 * @default true
	 */
	supportDemoLiveSwitcher?: boolean;
	/**
	 * Enables brackets for market orders.
	 * @default true
	 */
	supportMarketBrackets?: boolean;
	/**
	 * Enables symbol searching.
	 * @default false
	 */
	supportSymbolSearch?: boolean;
	/**
	 * Allows modifying existing [order duration](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/order-ticket.md#set-order-duration).
	 * @default false
	 */
	supportModifyDuration?: boolean;
	/**
	 * Allows modifying trailing stop orders.
	 * @default true
	 */
	supportModifyTrailingStop?: boolean;
	/**
	 * Allows margin.
	 * If `supportMargin` is set to `true`, you should call {@link IBrokerConnectionAdapterHost.marginAvailableUpdate} when the Trading Platform subscribes to margin available updates using {@link IBrokerWithoutRealtime.subscribeMarginAvailable}.
	 * @default false
	 */
	supportMargin?: boolean;
	/**
	 * Enables Profit & Loss calculations using last value.
	 * @default false
	 */
	calculatePLUsingLast?: boolean;
	/**
	 * Allows providing the estimated commission, fees, margin, and other order information before placing the order without actually placing it.
	 * This information will be displayed in the _Order confirmation_ dialog.
	 *
	 * This flag requires the {@link IBrokerTerminal.previewOrder} method to be implemented and `confirmId` parameter to be passed in the {@link IBrokerTerminal.placeOrder} method.
	 * Refer to [Enable order preview](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/order-ticket.md#add-custom-fields) for more information.
	 * @default false
	 */
	supportPlaceOrderPreview?: boolean;
	/**
	 * Allows providing the estimated commission, fees, margin, and other order information before modifying the order without actually modifying it.
	 * This information will be displayed in the _Order confirmation_ dialog.
	 *
	 * This flag requires the {@link IBrokerTerminal.previewOrder} method to be implemented and `confirmId` parameter to be passed in the {@link IBrokerTerminal.modifyOrder} method.
	 * Refer to [Enable order preview](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/order-ticket.md#add-custom-fields) for more information.
	 * @default false
	 */
	supportModifyOrderPreview?: boolean;
	/**
	 * Enables trading with leverage.
	 * If the flag is set to `true`, you should calculate the leverage using the {@link IBrokerWithoutRealtime.leverageInfo} method.
	 * @default false
	 */
	supportLeverage?: boolean;
	/**
	 * Displays a leverage button in the UI.
	 * Note that you should also enable the {@link BrokerConfigFlags.supportLeverage} flag.
	 * If `supportLeverageButton` is set to `true`, the leverage input field appears in the Order Ticket.
	 * Clicking the input field activates a dedicated Leverage Dialog.
	 * @default true
	 */
	supportLeverageButton?: boolean;
	/**
	 * Enables orders history.
	 * If `supportOrdersHistory` is set to `true`, the [Account Manager](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md) will have an additional tab: _Orders History_.
	 * This flag requires the {@link IBrokerTerminal.ordersHistory} method to be implemented.
	 * The method should return a list of orders with the `filled`, `cancelled`, and `rejected` statuses from previous trade sessions.
	 * @default false
	 */
	supportOrdersHistory?: boolean;
	/**
	 * Enables adding brackets to the existing order.
	 * @default true
	 */
	supportAddBracketsToExistingOrder?: boolean;
	/**
	 * Allows getting crypto balances for an account.
	 * Balances are displayed as the first table of the _Account Summary_ tab.
	 * Use the flag for crypto currencies only.
	 * This flag requires the {@link IBrokerConnectionAdapterHost.cryptoBalanceUpdate} method to be implemented.
	 * @default false
	 */
	supportBalances?: boolean;
	/**
	 * Closing a position cancels its brackets.
	 * @default false
	 */
	closePositionCancelsOrders?: boolean;
	/**
	 * `Stop Loss` and `Take Profit` are added or removed only together.
	 * @default false
	 */
	supportOnlyPairPositionBrackets?: boolean;
	/**
	 * Enables cryptocurrency trading (exchanging).
	 * This flag switches the Order Ticket to Crypto Exchange mode,
	 * which provides additional controls for entering the quantity in either the base or quote currency.
	 * @default false
	 */
	supportCryptoExchangeOrderTicket?: boolean;
	/**
	 * Enables displaying Profit & Loss values in instrument currency.
	 * @default false
	 */
	positionPLInInstrumentCurrency?: boolean;
	/**
	 * Enables partial position closing.
	 * This flag requires the {@link IBrokerTerminal.closePosition} method to be implemented.
	 * @default false
	 */
	supportPartialClosePosition?: boolean;
	/**
	 * Enables partial individual position closing.
	 * This flag requires the {@link IBrokerTerminal.closeIndividualPosition} method to be implemented.
	 * @default false
	 */
	supportPartialCloseIndividualPosition?: boolean;
	/**
	 * Modifies the confirmation dialog text for closing a bracket order.
	 * When set to `true`, the text explicitly states that cancelling a bracket order will also cancel its associated pair.
	 * When set to `false`, the text will include the ID of the singular bracket order being cancelled.
	 *
	 * Note that the library does not cancel orders itself.
	 * You should implement the {@link IBrokerTerminal.cancelOrder} or {@link IBrokerTerminal.cancelOrders } method.
	 * @default false
	 */
	supportCancellingBothBracketsOnly?: boolean;
	/**
	 * Enables crypto brackets.
	 * @default false
	 */
	supportCryptoBrackets?: boolean;
	/**
	 * Enables the _Notifications log_ tab in the [Account Manager](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/account-manager.md).
	 * @default true
	 */
	showNotificationsLog?: boolean;
	/**
	 * Whether stop orders should behave like Market-if-Touched in both directions.
	 * Enabling this flag prevents the check of stop price direction from the stop limit Order Ticket.
	 * @default false
	 */
	supportStopOrdersInBothDirections?: boolean;
	/**
	 * Whether stop-limit orders should behave like Limit-if-Touched in both directions.
	 * Enabling this flag prevents the check of stop price direction from the stop limit Order Ticket.
	 */
	supportStopLimitOrdersInBothDirections?: boolean;
	/**
	 * Whether the integration supports limit price validation in the order ticket to eliminate the possibility to place
	 * an order on the wrong side of the market that will most likely trigger and get filled immediately.
	 */
	supportStrictCheckingLimitOrderPrice?: boolean;
	/**
	 * Enables executions.
	 * This flag requires the {@link IBrokerTerminal.executions} method to be implemented.
	 * @default false
	 */
	supportExecutions?: boolean;
	/**
	 * Allows modifying order type.
	 * @default false
	 */
	supportModifyOrderType?: boolean;
	/**
	 * Enables closing of individual positions in FIFO order.
	 * @default false
	 */
	requiresFIFOCloseIndividualPositions?: boolean;
}
export interface BrokerCustomUI {
	/**
	 * Shows standard Order Ticket to create or modify an order and executes handler if Buy/Sell/Modify is pressed.
	 * @param  {OrderTemplate|Order} order - order to be placed or modified
	 * @param  {OrderTicketFocusControl} [focus] - Control to focus on when dialog is opened
	 */
	showOrderDialog?: (order: OrderTemplate | Order, focus?: OrderTicketFocusControl) => Promise<boolean>;
	/**
	 * Shows the Position Dialog
	 * @param  {Position|IndividualPosition} position - position to be placed or modified
	 * @param  {Brackets} brackets - brackets for the position
	 * @param  {OrderTicketFocusControl} [focus] - Control to focus on when dialog is opened
	 */
	showPositionDialog?: (position: Position | IndividualPosition, brackets: Brackets, focus?: OrderTicketFocusControl) => Promise<boolean>;
	/**
	 * Shows a confirmation dialog and executes handler if YES/OK is pressed.
	 * @param  {Order} order - order to be cancelled
	 */
	showCancelOrderDialog?: (order: Order) => Promise<boolean>;
	/**
	 * Shows the Close Position Dialog.
	 * @param  {Position} position - position to be closed
	 */
	showClosePositionDialog?: (position: Position) => Promise<boolean>;
}
export interface ChangeAccountSolution {
	/** id of a sub-account suitable for trading the symbol */
	changeAccount: AccountId;
}
export interface ChangeSymbolSolution {
	/** the symbol suitable for trading with current sub-account */
	changeSymbol: string;
}
export interface CheckboxFieldMetaInfo extends CustomFieldMetaInfoBase {
	/** @inheritDoc */
	inputType: "Checkbox";
	/** @inheritDoc */
	value: boolean;
	/** Does the field support modification */
	supportModify?: boolean;
	/** Help message for the field */
	help?: string;
}
export interface CryptoBalance {
	/** Symbol */
	symbol: string;
	/** Total balance */
	total: number;
	/** Available balance */
	available: number;
	/** Reserved balance */
	reserved?: number;
	/** Balance value */
	value?: number;
	/** Balance value's currency */
	valueCurrency?: string;
	/** Long name of Crypto */
	longName?: string;
	/** Bitcoin value of balance */
	btcValue?: number;
}
export interface CustomComboBoxItem {
	/** Combo box item text */
	text: string;
	/** Combo box item value */
	value: string;
}
export interface CustomComboBoxMetaInfo extends CustomInputFieldMetaInfo {
	/** @inheritDoc */
	inputType: "ComboBox";
	/** Items for the combo box input field */
	items: CustomComboBoxItem[];
}
export interface CustomFieldMetaInfoBase {
	/** Type of the input field */
	inputType: string;
	/** Input field ID */
	id: string;
	/** Title for the input field */
	title: string;
	/** Value of the field */
	value?: any;
	/** Should the input field value be saved to settings */
	saveToSettings?: boolean;
}
/**
 * Custom fields to be added to an object.
 */
export interface CustomFields {
	/** Custom field */
	[key: string]: any;
}
export interface CustomInputFieldMetaInfo extends CustomFieldMetaInfoBase {
	/** Prevent modification */
	preventModify?: boolean;
	/** Placeholder string for the field */
	placeHolder?: string;
	/** Validator function for the field */
	validator?: InputFieldValidator;
	/** Additional custom information */
	customInfo?: any;
}
/**
 * An object that contains the results of broker specific user inputs (for example a digital signature).
 */
export interface CustomInputFieldsValues {
	[fieldId: string]: TextWithCheckboxValue | boolean | string | any;
}
export interface CustomTableElementFormatter<T extends TableFormatterInputValues = TableFormatterInputValues> {
	/** Custom formatter name */
	name: FormatterName;
	/** Formatter to generate HTML element */
	formatElement?: CustomTableFormatElementFunction<T>;
	/** Formatter to generate text. Return an empty string if you don't need to display this */
	formatText: TableFormatTextFunction<T>;
	/** Allow usage of priceFormatter */
	isPriceFormatterNeeded?: boolean;
}
/**
 * Depth of Market (Order Book) Data
 */
export interface DOMData {
	/**
	 * Whether the Depth of Market data is a snapshot (has the full set of depth data).
	 * - If `true` then the data contains the full set of depth data.
	 * - If `false` then data only contains updated levels.
	 */
	snapshot: boolean;
	/** Ask order levels (must be sorted by `price` in ascending order) */
	asks: DOMLevel[];
	/** Bid order levels (must be sorted by `price` in ascending order) */
	bids: DOMLevel[];
}
/**
 * Depth of Market Level
 */
export interface DOMLevel {
	/** Price for DOM level */
	price: number;
	/** Volume for DOM level */
	volume: number;
}
export interface DefaultContextMenuActionsParams {
}
export interface DefaultDropdownActionsParams {
	/** Show trading properties */
	tradingProperties?: boolean;
	/** Restore confirmations */
	restoreConfirmations?: boolean;
}
export interface ErrorFormatterParseResult extends FormatterParseResult {
	/** Optional message when there's an error while parsing */
	error?: string;
	/** @inheritDoc */
	res: false;
}
/**
 * Describes a single execution.
 * Execution is when a buy or sell order is completed for a financial instrument.
 */
export interface Execution extends CustomFields {
	/** Symbol name */
	symbol: string;
	/** Execution price */
	price: number;
	/** Execution Quantity */
	qty: number;
	/** Execution Side */
	side: Side;
	/** Time (unix timestamp in milliseconds) */
	time: number;
	/** Commission amount for executed trade */
	commission?: number;
	/** Net amount for executed trade */
	netAmount?: number;
}
export interface FormatterParseResult {
	/** Returns if the formatter support parsing */
	res: boolean;
}
export interface IBoxedValue<T> extends IBoxedValueReadOnly<T> {
	/**
	 * Set boxed value
	 * @param  {T} value - value to be set
	 */
	setValue(value: T): void;
}
export interface IBoxedValueReadOnly<T> {
	/** Value */
	value(): T;
}
export interface IBrokerAccountInfo {
	accountsMetainfo(): Promise<AccountMetainfo[]>;
	currentAccount(): AccountId;
	setCurrentAccount?(id: AccountId): void;
}
export interface IBrokerCommon {
	/**
	 * Chart can have a sub-menu `Trading` in the context menu. This method should return an array of {@link ActionMetaInfo} elements, each of them representing one context menu item.
	 * @param  {TradeContext} context - context object passed by a browser
	 * @param  {DefaultContextMenuActionsParams} options? - default options for the context menu action parameters
	 */
	chartContextMenuActions(context: TradeContext, options?: DefaultContextMenuActionsParams): Promise<ActionMetaInfo[]>;
	/**
	 * This function is required for the Floating Trading Panel.
	 * The ability to trade via the panel depends on the result of this function: `true` or `false`.
	 * You don't need to implement this method if all symbols can be traded.
	 *
	 * If you want to show a custom message with the reason why the symbol cannot be traded then you can return an object `IsTradableResult`.
	 * @param  {string} symbol - symbol identifier
	 */
	isTradable(symbol: string): Promise<boolean | IsTradableResult>;
	/**
	 * Connection status for the Broker API.
	 *
	 * You don't need to return values other than `1` (`Connected`) typically since the broker is already connected when you create the widget.
	 * You can use it if you want to display a spinner in the bottom panel while the data is being loaded.
	 */
	connectionStatus(): ConnectionStatus;
	/**
	 * Called by Trading Platform to request orders
	 */
	orders(): Promise<Order[]>;
	/**
	 * This method is called by the Trading Platform to request orders history.
	 * It is expected that returned orders will have a final status (`rejected`, `filled`, `cancelled`).
	 *
	 * This method is optional. If you don't support orders history, please set the {@link BrokerConfigFlags.supportOrdersHistory} flag to `false`.
	 */
	ordersHistory?(): Promise<Order[]>;
	/**
	 * Called by Trading Platform to request positions.
	 * Required if the {@link BrokerConfigFlags.supportPositions} flag is set to `true`.
	 */
	positions?(): Promise<Position[]>;
	/**
	 * Called by Trading Platform to request individual positions.
	 * Required if the {@link BrokerConfigFlags.supportPositionNetting} flag is set to `true`.
	 */
	individualPositions?(): Promise<IndividualPosition[]>;
	/**
	 * Called by Trading Platform to request executions for the specified symbol
	 * @param  {string} symbol - symbol identifier
	 */
	executions(symbol: string): Promise<Execution[]>;
	/**
	 * Called by the internal Order dialog, DOM panel, and floating trading panel to get symbol information.
	 * @param  {string} symbol - symbol identifier
	 */
	symbolInfo(symbol: string): Promise<InstrumentInfo>;
	/**
	 * This function should return the information that will be used to build an Account manager.
	 */
	accountManagerInfo(): AccountManagerInfo;
	/**
	 * Provide a custom price formatter for the specified symbol.
	 * @param  {string} symbol - symbol identifier
	 * @param  {boolean} alignToMinMove - align formatted number to the minimum movement amount of the symbol
	 */
	formatter?(symbol: string, alignToMinMove: boolean): Promise<INumberFormatter>;
	/**
	 * Provide a custom spread formatter for the specified symbol.
	 * @param  {string} symbol - symbol identifier
	 */
	spreadFormatter?(symbol: string): Promise<INumberFormatter>;
	/**
	 * Provide a custom quantity formatter for the specified symbol.
	 * @param  {string} symbol - symbol identifier
	 */
	quantityFormatter?(symbol: string): Promise<INumberFormatter>;
	/**
	 * Implement this method if you want to [add custom fields](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/order-ticket.md#add-custom-fields) to the standard Order Ticket.
	 *
	 * Use the `symbol` parameter to return customization options for a particular symbol.
	 * @param  {string} symbol - symbol identifier
	 */
	getOrderDialogOptions?(symbol: string): Promise<OrderDialogOptions | undefined>;
	/**
	 * Implement this method if you want to customize the position dialog.
	 */
	getPositionDialogOptions?(): PositionDialogOptions | undefined;
	/**
	 * Implement this method if you want to have custom options available for different symbols.
	 * @param  {string} symbol - symbol identifier
	 */
	getSymbolSpecificTradingOptions?(symbol: string): Promise<SymbolSpecificTradingOptions | undefined>;
}
export interface IBrokerConnectionAdapterFactory {
	/** Creates a Delegate object */
	createDelegate<T extends Function>(): IDelegate<T>;
	/** Creates a WatchedValue object */
	createWatchedValue<T>(value?: T): IWatchedValue<T>;
	/**
	 * Creates a price formatter.
	 * @param priceScale - Defines the number of decimal places. It is `10^number-of-decimal-places`. If a price is displayed as `1.01`, `pricescale` is `100`; If it is displayed as `1.005`, `pricescale` is `1000`.
	 * @param minMove - The amount of price precision steps for 1 tick. For example, since the tick size for U.S. equities is `0.01`, `minmov` is 1. But the price of the E-mini S&P futures contract moves upward or downward by `0.25` increments, so the `minmov` is `25`.
	 * @param fractional - For common prices, is `false` or it can be skipped. For more information on fractional prices, refer to [Fractional format](https://www.tradingview.com/charting-library-docs/latest/connecting_data/Symbology.md#fractional-format).
	 * @param minMove2 - For common prices, is `0` or it can be skipped.
	 * @param variableMinTick - For common prices, is `string` (for example, `0.01 10 0.02 25 0.05`) or it can be skipped. For more information, refer to [Variable tick size](https://www.tradingview.com/charting-library-docs/latest/connecting_data/Symbology.md#variable-tick-size).
	 */
	createPriceFormatter(priceScale?: number, minMove?: number, fractional?: boolean, minMove2?: number, variableMinTick?: string): IPriceFormatter;
}
/**
 * The Trading Host is an API for interaction between the Broker API and the library code related to trading.
 * Its main purpose is to receive information from your backend server where trading logic is implemented and provide updates to the library.
 * Refer to the [Core trading concepts](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/trading-concepts/trading-concepts.md) article for more information.
 */
export interface IBrokerConnectionAdapterHost {
	/** Broker Connection Adapter Factory object */
	factory: IBrokerConnectionAdapterFactory;
	/**
	 * Generates and returns the default value formatter for the symbol
	 * @param  {string} symbol - symbol identifier
	 * @param  {boolean} alignToMinMove - whether the formatted number should be aligned to minimum movement for the symbol
	 */
	defaultFormatter(symbol: string, alignToMinMove: boolean): Promise<INumberFormatter>;
	/**
	 * Generates and returns a number formatter with the desired decimal places
	 * @param  {number} [decimalPlaces] - decimal places
	 */
	numericFormatter(decimalPlaces: number): Promise<INumberFormatter>;
	/**
	 * Generates and returns a quantity formatter with the desired decimal places
	 * @param  {number} [decimalPlaces] - decimal places
	 */
	quantityFormatter(decimalPlaces?: number): Promise<INumberFormatter>;
	/**
	 * Provides default buy/sell, show properties actions to be returned as a default by {@link IBrokerCommon.chartContextMenuActions}.
	 * @param  {TradeContext} context - trade context
	 * @param  {DefaultContextMenuActionsParams} [params] - optional parameters
	 */
	defaultContextMenuActions(context: TradeContext, params?: DefaultContextMenuActionsParams): Promise<ActionMetaInfo[]>;
	/**
	 * Provides default dropdown list of actions. You can use default actions in {@link IBrokerConnectionAdapterHost.setButtonDropdownActions}
	 * @param  {Partial<DefaultDropdownActionsParams>} [options] - options for the dropdown menu actions
	 */
	defaultDropdownMenuActions(options?: Partial<DefaultDropdownActionsParams>): ActionMetaInfo[];
	/** Returns whether the buy/sell buttons are visible or not. */
	sellBuyButtonsVisibility(): IWatchedValue<boolean> | null;
	/** Returns whether DOM Panel is visible or not. */
	domPanelVisibility(): IWatchedValue<boolean> | null;
	/** Returns whether the order panel is visible or not. */
	orderPanelVisibility(): IWatchedValue<boolean> | null;
	/** Returns if orders can be sent to the broker without showing the order ticket. */
	silentOrdersPlacement(): IWatchedValue<boolean>;
	/**
	 * Patch changes into the current broker configuration
	 * @param  {Partial<BrokerConfigFlags>} config - partial configuration changes to apply
	 */
	patchConfig(config: Partial<BrokerConfigFlags>): void;
	/**
	 * Set expiration durations
	 * @param  {OrderDurationMetaInfo[]} durations - Expiration options for orders
	 */
	setDurations(durations: OrderDurationMetaInfo[]): void;
	/**
	 * Call this method to notify the chart that it needs to update information after an order is added or changed.
	 * @param  {Order} order - order which was added or changed
	 */
	orderUpdate(order: Order): void;
	/**
	 * Call this method when an order is not changed, but the fields that you added to the order object to display in the Account Manager have changed.
	 * It should be used only if you want to display custom fields in the Account Manager.
	 * @param  {string} id - order id
	 * @param  {Partial<Order>} orderChanges - changes made to the order object
	 */
	orderPartialUpdate(id: string, orderChanges: Partial<Order>): void;
	/**
	 * Call this method when a position is added or changed.
	 * @param  {Position} position - position which was added or changed
	 * @param  {boolean} [isHistoryUpdate] - whether the change is a history update
	 */
	positionUpdate(position: Position, isHistoryUpdate?: boolean): void;
	/**
	 * Call this method when a position is not changed, but the fields that you added to the position object to display in the Account Manager have changed.
	 * It should be used only if you want to display custom fields in the Account Manager.
	 * @param  {string} id - id of the position
	 * @param  {Partial<Position>} positionChanges - changes to the position object
	 */
	positionPartialUpdate(id: string, positionChanges: Partial<Position>): void;
	/**
	 * Call this method when an individual position is added or changed.
	 * @param  {IndividualPosition} individualPosition - updated individual position
	 * @param  {boolean} [isHistoryUpdate] - whether the change is a history update
	 */
	individualPositionUpdate(individualPosition: IndividualPosition, isHistoryUpdate?: boolean): void;
	/**
	 * Call this method when an individual position has not changed, but fields that you added to the individual position object to display in the Account Manager have changed.
	 * @param  {string} id - ID of the updated individual position
	 * @param  {Partial<IndividualPosition>} changes - changes to the individual position object
	 */
	individualPositionPartialUpdate(id: string, changes: Partial<IndividualPosition>): void;
	/**
	 * Call this method when an execution is added.
	 * @param  {Execution} execution - execution which was added
	 */
	executionUpdate(execution: Execution): void;
	/**
	 * Call this method when user account has been changed synchronously. The terminal will re-request all displayed information.
	 */
	currentAccountUpdate(): void;
	/**
	 * Trading quote realtime update
	 * @param  {string} symbol - symbol identifier
	 * @param  {TradingQuotes} data - realtime updated data for the symbol quotes
	 */
	realtimeUpdate(symbol: string, data: TradingQuotes): void;
	/**
	 * Call this method when a broker connection has received a PL update.
	 * Use this method when the {@link BrokerConfigFlags.supportPLUpdate} flag is set to `true` in {@link SingleBrokerMetaInfo.configFlags}.
	 * @param  {string} positionId - ID of the position
	 * @param  {number} pl - updated profit / loss value
	 */
	plUpdate(positionId: string, pl: number): void;
	/**
	 * Call this method when a broker connection has a `pipValue` update.
	 * The library subscribes to `pipValue` updates using {@link IBrokerWithoutRealtime.subscribePipValue}.
	 * @param  {string} symbol - symbol with updated pip values
	 * @param  {PipValues} pipValues - updated pip values
	 */
	pipValueUpdate(symbol: string, pipValues: PipValues): void;
	/**
	 * Call this method when a broker connection has received an individual position PL update.
	 * @param  {string} individualPositionId - ID of the individual position
	 * @param  {number} pl - updated profit / loss for the individual position
	 */
	individualPositionPLUpdate(individualPositionId: string, pl: number): void;
	/**
	 * Call this method when a broker connection has received an equity update. This method is required by the standard Order Dialog to calculate risks.
	 * @param  {number} equity - updated equity
	 */
	equityUpdate(equity: number): void;
	/**
	 * Call this method when a broker connection has received a margin available update.
	 * This method is required by the standard Order Dialog to display the margin meter.
	 * This method should be used when {@link BrokerConfigFlags.supportMargin} is set to `true` in {@link SingleBrokerMetaInfo.configFlags}.
	 * The Trading Platform subscribes to margin available updates using {@link IBrokerWithoutRealtime.subscribeMarginAvailable}.
	 * @param  {number} marginAvailable - updated available margin
	 */
	marginAvailableUpdate(marginAvailable: number): void;
	/**
	 * Call this method when a broker connection has received a balance update.
	 * This method is required by the crypto Order Dialog.
	 * It should be implemented when the {@link BrokerConfigFlags.supportBalances} flag is set to `true` in {@link SingleBrokerMetaInfo.configFlags}.
	 * @param  {string} symbol - symbol ID
	 * @param  {CryptoBalance} balance - updated crypto balance
	 */
	cryptoBalanceUpdate(symbol: string, balance: CryptoBalance): void;
	/**
	 * Update the Depth of Market for the specified symbol
	 * @param  {string} symbol - symbol identifier
	 * @param  {DOMData} equity - Depth of market data
	 */
	domUpdate(symbol: string, equity: DOMData): void;
	/**
	 * Sets the quantity for a given symbol.
	 * @param  {string} symbol - symbol
	 * @param  {number} quantity - quantity to update
	 */
	setQty(symbol: string, quantity: number): void;
	/**
	 * Returns the quantity for a given symbol.
	 * @param  {string} symbol - symbol
	 * @return  {Promise<number>} - quantity for the given symbol
	 */
	getQty(symbol: string): Promise<number>;
	/**
	 * Adds a callback to be executed whenever there's a change of quantity for a given symbol.
	 *
	 * It's the user's responsibility to manage the unsubscription of any added listener
	 *
	 * @param  {string} symbol - symbol to which the callback will be linked to
	 * @param  {SuggestedQtyChangedListener} listener - callback
	 */
	subscribeSuggestedQtyChange(symbol: string, listener: SuggestedQtyChangedListener): void;
	/**
	 * Remove a previously added callback from the list.
	 * @param  {string} symbol - symbol to remove the callback from
	 * @param  {SuggestedQtyChangedListener} listener - callback to be removed
	 */
	unsubscribeSuggestedQtyChange(symbol: string, listener: SuggestedQtyChangedListener): void;
	/**
	 * Shows the order dialog
	 * @param  {T extends PreOrder} order - order to show in the dialog
	 * @param  {OrderTicketFocusControl} focus? - input control to focus on when dialog is opened
	 */
	showOrderDialog?<T extends PreOrder>(order: T, focus?: OrderTicketFocusControl): Promise<boolean>;
	/**
	 * Shows notification message
	 * @param  {string} title - notification title
	 * @param  {string} text - notification content
	 * @param  {NotificationType} notificationType? - type of notification (default: NotificationType.Error)
	 */
	showNotification(title: string, text: string, notificationType?: NotificationType): void;
	/**
	 * Shows the cancel order dialog for specified order
	 * @param  {string} orderId - id of order to potentially cancel
	 * @param  {()=>Promise<void>} handler - cancel order confirmation handler (called when order should be cancelled)
	 */
	showCancelOrderDialog(orderId: string, handler: () => Promise<void>): Promise<void>;
	/**
	 * Shows the cancel order dialog for multiple orders
	 * @param  {string} symbol - symbol for which to cancel orders
	 * @param  {Side} side - side of the order
	 * @param  {number} qty - quantity of the order
	 * @param  {()=>Promise<void>} handler - cancel orders confirmation handler (called when orders should be cancelled)
	 */
	showCancelMultipleOrdersDialog(symbol: string, side: Side, qty: number, handler: () => Promise<void>): Promise<void>;
	/**
	 * Shows the cancel brackets dialog
	 * @param  {string} orderId - id of order
	 * @param  {()=>Promise<void>} handler - cancel brackets confirmation handler (called when brackets should be cancelled)
	 */
	showCancelBracketsDialog(orderId: string, handler: () => Promise<void>): Promise<void>;
	/**
	 * Shows the cancel brackets dialog for multiple brackets
	 * @param  {string} orderId - id of order
	 * @param  {()=>Promise<void>} handler - cancel brackets confirmation handler (called when brackets should be cancelled)
	 */
	showCancelMultipleBracketsDialog(orderId: string, handler: () => Promise<void>): Promise<void>;
	/**
	 * Shows reverse position dialog
	 * @param  {string} position - position to be reversed
	 * @param  {()=>Promise<boolean>} handler - reverse position confirmation handler (called when the position should be reversed)
	 */
	showReversePositionDialog(position: string, handler: () => Promise<boolean>): Promise<boolean>;
	/**
	 * Shows the position brackets dialog
	 * @param  {Position|IndividualPosition} position - position or individual position
	 * @param  {Brackets} brackets - brackets for the position or individual position
	 * @param  {OrderTicketFocusControl} focus - input control to focus on when dialog is opened
	 */
	showPositionBracketsDialog(position: Position | IndividualPosition, brackets: Brackets, focus: OrderTicketFocusControl): Promise<boolean>;
	/**
	 * Activate bottom widget
	 */
	activateBottomWidget(): Promise<void>;
	/**
	 * Shows trading properties
	 */
	showTradingProperties(): void;
	/**
	 * Returns symbol `minTick`.
	 * @param  {string} symbol - symbol identifier
	 */
	getSymbolMinTick(symbol: string): Promise<number>;
	/**
	 * Displays a message dialog to a user.
	 * @param  {string} title - title of the message dialog
	 * @param  {string} text - message
	 * @param  {boolean} textHasHTML? - whether message text contains HTML
	 */
	showMessageDialog(title: string, text: string, textHasHTML?: boolean): void;
	/**
	 * Displays a confirmation dialog to a user and returns a Promise to the result.
	 * @param  {string} title - title of the confirmation dialog
	 * @param  {string|string[]} content - content for the dialog
	 * @param  {string} [mainButtonText] - text for the main button (`true` result)
	 * @param  {string} [cancelButtonText] - text for the cancel button (`false` result)
	 * @param  {boolean} [showDisableConfirmationsCheckbox] - show disable confirmations checkbox within the dialog
	 */
	showConfirmDialog(title: string, content: string | string[], mainButtonText?: string, cancelButtonText?: string, showDisableConfirmationsCheckbox?: boolean): Promise<boolean>;
	/**
	 * Displays a simple confirmation dialog to a user and returns a Promise to the result.
	 * @param  {string} title - title of the confirmation dialog
	 * @param  {string|string[]} content - content for the dialog
	 * @param  {string} [mainButtonText] - text for the main button (`true` result)
	 * @param  {string} [cancelButtonText] - text for the cancel button (`false` result)
	 * @param  {boolean} [showDisableConfirmationsCheckbox] - show disable confirmations checkbox within the dialog
	 */
	showSimpleConfirmDialog(title: string, content: string | string[], mainButtonText?: string, cancelButtonText?: string, showDisableConfirmationsCheckbox?: boolean): Promise<boolean>;
}
export interface IBrokerTerminal extends IBrokerWithoutRealtime {
	/**
	 * Library is requesting that realtime updates should be supplied for this symbol.
	 * @param  {string} symbol - symbol identifier
	 */
	subscribeRealtime(symbol: string): void;
	/**
	 * Library is notifying that realtime updates are no longer required for this symbol.
	 * @param  {string} symbol - symbol identifier
	 */
	unsubscribeRealtime(symbol: string): void;
}
/**
 * The Broker API is a key component that enables trading.
 * Its main purpose is to connect TradingView charts with your trading logic.
 * Refer to the [Core trading concepts](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/trading-concepts/trading-concepts.md) article for more information.
 */
export interface IBrokerWithoutRealtime extends IBrokerCommon, IBrokerAccountInfo {
	/**
	 * Library is requesting that realtime DOM (Depth of Market) updates should be supplied for this symbol
	 * @param  {string} symbol - symbol identifier
	 */
	subscribeDOM?(symbol: string): void;
	/**
	 * Library is notifying that realtime DOM (Depth of Market) updates are no longer required for this symbol.
	 * @param  {string} symbol - symbol identifier
	 */
	unsubscribeDOM?(symbol: string): void;
	/**
	 * Method is called when a user wants to place an order.
	 * Order is pre-filled with partial or complete information.
	 * This function returns an object with the order ID.
	 * To enable order preview before placing it, set {@link BrokerConfigFlags.supportPlaceOrderPreview} to `true`.
	 * @param  {PreOrder} order - order information
	 * @param  {string} [confirmId] - is passed if the `supportPlaceOrderPreview` configuration flag is on.
	 * @returns PlaceOrderResult, which should include an `orderId`
	 */
	placeOrder(order: PreOrder, confirmId?: string): Promise<PlaceOrderResult>;
	/**
	 * Returns estimated commission, fees, margin, and other information for the order without it actually being placed.
	 * The method is called if the {@link BrokerConfigFlags.supportPlaceOrderPreview} or {@link BrokerConfigFlags.supportModifyOrderPreview} configuration flag is on.
	 * @param  {PreOrder} order - order information
	 */
	previewOrder?(order: PreOrder): Promise<OrderPreviewResult>;
	/**
	 * Method is called when a user wants to modify an existing order.
	 *
	 * Note that the library expects you to call the {@link IBrokerConnectionAdapterHost.orderUpdate} method right afterwards.
	 * Otherwise, the library will return a [timeout issue](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/common-issues.md#timeout-issue).
	 *
	 * To enable order preview before modifying it, set {@link BrokerConfigFlags.supportModifyOrderPreview} to `true`.
	 * @param  {Order} order - order information
	 * @param  {string} [confirmId] - is passed if `supportModifyOrderPreview` configuration flag is on.
	 */
	modifyOrder(order: Order, confirmId?: string): Promise<void>;
	/**
	 * This method is called to cancel a single order with the given `id`.
	 *
	 * Note that the library expects you to call the {@link IBrokerConnectionAdapterHost.orderUpdate} method right afterwards.
	 * @param  {string} orderId - ID for the order to cancel
	 */
	cancelOrder(orderId: string): Promise<void>;
	/**
	 * This method is called to cancel multiple orders for a `symbol` and `side`.
	 * The `ordersIds` parameter should contain the list of order ids to be cancelled.
	 *
	 * Note that the library expects you to call the {@link IBrokerConnectionAdapterHost.orderUpdate} method right afterwards.
	 * @param  {string} symbol - symbol identifier
	 * @param  {Side|undefined} side - order side
	 * @param  {string[]} ordersIds - ids already collected by `symbol` and `side`
	 */
	cancelOrders(symbol: string, side: Side | undefined, ordersIds: string[]): Promise<void>;
	/**
	 * This method is called if the {@link BrokerConfigFlags.supportNativeReversePosition} configuration flag is on.
	 * It allows reversing the position by ID.
	 *
	 * Note that the library expects you to call the {@link IBrokerConnectionAdapterHost.positionUpdate} method right afterwards.
	 * Otherwise, the library will return a [timeout issue](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/common-issues.md#timeout-issue).
	 * @param  {string} positionId - position
	 */
	reversePosition?(positionId: string): Promise<void>;
	/**
	 * This method is called if the {@link BrokerConfigFlags.supportClosePosition} or {@link BrokerConfigFlags.supportPartialClosePosition} configuration flag is on.
	 * It allows closing the position by ID.
	 *
	 * Note that the library expects you to call the {@link IBrokerConnectionAdapterHost.positionUpdate} method right afterwards.
	 * Otherwise, the library will return a [timeout issue](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/common-issues.md#timeout-issue).
	 * @param  {string} positionId - Position ID.
	 * @param  {number} [amount] - The amount is specified if `supportPartialClosePosition` is `true` and the user wants to close only part of the position.
	 */
	closePosition?(positionId: string, amount?: number): Promise<void>;
	/**
	 * This method is called if the {@link BrokerConfigFlags.supportCloseIndividualPosition} or {@link BrokerConfigFlags.supportPartialCloseIndividualPosition} configuration flag is on.
	 * It allows closing the individual position by ID.
	 *
	 * Note that the library expects you to call the {@link IBrokerConnectionAdapterHost.positionUpdate} method right afterwards.
	 * Otherwise, the library will return a [timeout issue](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/common-issues.md#timeout-issue).
	 * @param  {string} individualPositionId - Individual position ID.
	 * @param  {number} [amount] - The amount is specified if `supportPartialCloseIndividualPosition` is `true` and the user wants to close only part of the individual position.
	 */
	closeIndividualPosition?(individualPositionId: string, amount?: number): Promise<void>;
	/**
	 * This method is called if the {@link BrokerConfigFlags.supportPositionBrackets} configuration flag is on.
	 * It shows a dialog that enables take-profit and stop-loss editing.
	 *
	 * Note that the library expects you to call the {@link IBrokerConnectionAdapterHost.positionUpdate} method right afterwards.
	 * @param  {string} positionId - is an ID of an existing position to be modified
	 * @param  {Brackets} brackets - new Brackets to be set for the position
	 * @param  {CustomInputFieldsValues} [customFields] - custom fields to display in the dialog
	 */
	editPositionBrackets?(positionId: string, brackets: Brackets, customFields?: CustomInputFieldsValues): Promise<void>;
	/**
	 * This method is called if the {@link BrokerConfigFlags.supportIndividualPositionBrackets} configuration flag is on.
	 * It displays a dialog that enables take-profit and stop-loss editing.
	 *
	 * Note that the library expects you to call the {@link IBrokerConnectionAdapterHost.positionUpdate} method right afterwards.
	 * @param  {string} individualPositionId - ID of existing individual position to be modified
	 * @param  {Brackets} brackets - new Brackets to be set for the individual position
	 */
	editIndividualPositionBrackets?(individualPositionId: string, brackets: Brackets): Promise<void>;
	/**
	 * This method is called to receive leverageInfo from the broker.
	 * @param  {LeverageInfoParams} leverageInfoParams - information about the specific symbol to provide leverage information for
	 */
	leverageInfo?(leverageInfoParams: LeverageInfoParams): Promise<LeverageInfo>;
	/**
	 * This method is called to send user's leverage value to the broker. The value should be verified and corrected on the broker's side if required, and sent back in the response.
	 * @param  {LeverageSetParams} leverageSetParams - `leverageSetParams` is an object similar to {@link leverageInfoParams}, but contains an additional `leverage: number` field, which holds the leverage value set by the user.
	 */
	setLeverage?(leverageSetParams: LeverageSetParams): Promise<LeverageSetResult>;
	/**
	 * This method is called to receive {@link LeveragePreviewResult} object which holds messages about the leverage value set by the user.
	 * @param  {LeverageSetParams} leverageSetParams - `leverageSetParams` is an object similar to {@link leverageInfoParams}, but contains an additional `leverage: number` field, which holds the leverage value set by the user.
	 */
	previewLeverage?(leverageSetParams: LeverageSetParams): Promise<LeveragePreviewResult>;
	/**
	 * The method should be implemented if you use the standard Order dialog and support stop loss. Equity is used to calculate Risk in Percent.
	 *
	 * Once this method is called the broker should provide equity (Balance + P/L) updates via {@link IBrokerConnectionAdapterHost.equityUpdate} method.
	 */
	subscribeEquity?(): void;
	/**
	 * The method should be implemented if you use the standard Order dialog and want to show the margin meter.
	 *
	 * Once this method is called the broker should provide margin available updates via {@link IBrokerConnectionAdapterHost.marginAvailableUpdate} method.
	 * @param  {string} symbol - symbol identifier
	 */
	subscribeMarginAvailable?(symbol: string): void;
	/**
	 * The method should be implemented if you use a standard Order dialog.
	 * `pipValues` is displayed in the Order info and it is used to calculate the Trade Value and risks.
	 * If this method is not implemented then `pipValue` from the `symbolInfo` is used in the order panel/dialog.
	 *
	 * Once this method is called the broker should provide `pipValue` updates via {@link IBrokerConnectionAdapterHost.pipValueUpdate} method.
	 * @param  {string} symbol - symbol identifier
	 */
	subscribePipValue?(symbol: string): void;
	/**
	 * The method should be implemented if you use a standard Order dialog and implement `subscribePipValue`.
	 *
	 * Once this method is called the broker should stop providing `pipValue` updates.
	 * @param  {string} symbol - symbol identifier
	 */
	unsubscribePipValue?(symbol: string): void;
	/**
	 * The method should be implemented if you use the standard Order dialog want to show the margin meter.
	 *
	 * Once this method is called the broker should stop providing margin available updates.
	 * @param  {string} symbol - symbol identifier
	 */
	unsubscribeMarginAvailable?(symbol: string): void;
	/**
	 * The method should be implemented if you use the standard Order dialog and support stop loss.
	 *
	 * Once this method is called the broker should stop providing equity updates.
	 */
	unsubscribeEquity?(): void;
}
export interface IDelegate<TFunc extends Function> extends ISubscription<TFunc> {
	/** Fire (Evoke) */
	fire: TFunc;
}
/** Definition of a formatter */
export interface IFormatter<T> {
	/** Whatever the input type, formats the data following a certain logic and return that value as a string  */
	format(value?: T): string;
	/** Check if the input value satisfies the logic and return either an error or the result of the parsing  */
	parse?(value: string): ErrorFormatterParseResult | SuccessFormatterParseResult<T>;
}
/** Specific formatter for number */
export interface INumberFormatter extends IFormatter<number> {
	/**
	 * Formatter for a price change
	 * @param currentPrice - current price
	 * @param prevPrice - previous price
	 */
	formatChange?(currentPrice: number, prevPrice: number): string;
}
export interface IObservable<T> {
	/**
	 * Subscribe to changes
	 * @param  {(value:T)=>void} callback - callback function to be evoked when observed value changes
	 */
	subscribe(callback: (value: T) => void): void;
	/**
	 * Unsubscribe from changes
	 * @param  {(value:T)=>void} callback - callback function to be unsubscribed
	 */
	unsubscribe(callback: (value: T) => void): void;
}
export interface IObservableValue<T> extends IBoxedValue<T>, IObservable<T> {
}
export interface IObservableValueReadOnly<T> extends IBoxedValueReadOnly<T>, IObservable<T> {
}
/**
 * Specific formatter for numbers
 */
export interface IPriceFormatter extends ISymbolValueFormatter {
	/**
	 * Price Formatter
	 * @param  {number} price - price
	 * @param  {boolean} signPositive? - add plus sign to result string.
	 * @param  {number} [tailSize] - add `tailSize` digits to fractional part of result string
	 * @param  {boolean} [signNegative] - add minus sign to result string.
	 * @param  {boolean} [useRtlFormat] - Use Right to left format
	 * @param  {boolean} [cutFractionalByPrecision] - cuts price by priceScalePrecision, without rounding.
	 * @returns formatted price
	 */
	format(price: number, signPositive?: boolean, tailSize?: number, signNegative?: boolean, useRtlFormat?: boolean, cutFractionalByPrecision?: boolean): string;
}
/**
 * A subscription. Used to subscribe callbacks to events.
 */
export interface ISubscription<TFunc extends Function> {
	/**
	 * Subscribe a callback function to this event. Subscribed callbacks are called when the event fires.
	 *
	 * @param obj Object used as the `this` value bound to the callback function.
	 * @param member Function called when the event is fired.
	 * @param singleshot `true` if the subscription should be automatically removed after the first time it is fired.
	 *
	 * @example
	 * ```
	 * // Log 'Series data loaded!' to the console the next time the data loaded event fires and then unsubscribe automatically.
	 * seriesApi.onDataLoaded().subscribe(null, () => { console.log('Series data loaded!'); }, true);
	 * ```
	 * Subscribe to an event within a class. Manually unsubscribe when some condition is true.
	 * ```
	 * class Example {
	 *   constructor(seriesApi) {
	 *     this._seriesApi = seriesApi;
	 *     this._seriesApi.onDataLoaded().subscribe(this, this._onDataLoaded);
	 *   }
	 *
	 *   _onDataLoaded() {
	 *     // Do something in response to the event.
	 *
	 *     if (someUnsubscribeCondition) {
	 *       this._seriesApi.onDataLoaded().unsubscribe(this, this._onDataLoaded);
	 *     }
	 *   }
	 * }
	 * ```
	 */
	subscribe(obj: object | null, member: TFunc, singleshot?: boolean): void;
	/**
	 * Unsubscribe a previously subscribed callback function.
	 * It is important that the `obj` and `member` arguments are the same as the ones passed when the callback was subscribed.
	 *
	 * @param obj Object passed as the `this` value when the callback was subscribed.
	 * @param member Function subscribed to the event.
	 */
	unsubscribe(obj: object | null, member: TFunc): void;
	/**
	 * Unsubscribe all callbacks that were subscribed with the same `obj` value.
	 *
	 * @param obj `obj` value passed when the callbacks were subscribed.
	 * @example
	 * ```
	 * // Unsubscribe all three callback functions at once in the `destroy` method.
	 * class Example {
	 *   constructor(seriesApi) {
	 *     this._seriesApi = seriesApi;
	 *     this._seriesApi.onDataLoaded().subscribe(this, this._callback1);
	 *     this._seriesApi.onDataLoaded().subscribe(this, this._callback2);
	 *     this._seriesApi.onDataLoaded().subscribe(this, this._callback3);
	 *   }
	 *
	 *   destroy() {
	 *     this._seriesapi.onDataLoaded().unsubscribeAll(this);
	 *   }
	 *
	 *   _callback1() { ... }
	 *   _callback2() { ... }
	 *   _callback3() { ... }
	 * }
	 * ```
	 */
	unsubscribeAll(obj: object | null): void;
}
export interface ISymbolValueFormatter {
	/** Default formatter function used to assign the correct sign (+ or -) to a number  */
	format(price: number, signPositive?: boolean): string;
	/**
	 * Formatter for a price change
	 * @param currentPrice - current price
	 * @param prevPrice - previous price
	 * @param signPositive - is the sign of the number positive
	 */
	formatChange?(currentPrice: number, prevPrice: number, signPositive?: boolean): string;
}
export interface IWatchedValue<T> extends IWatchedValueReadonly<T>, IObservableValue<T> {
	/**
	 * Set value for the watched value
	 * @param  {T} value - value to set
	 * @param  {boolean} forceUpdate? - force an update
	 */
	setValue(value: T, forceUpdate?: boolean): void;
	/** @inheritDoc */
	subscribe(callback: WatchedValueCallback<T>, options?: WatchedValueSubscribeOptions): void;
	/** @inheritDoc */
	unsubscribe(callback?: WatchedValueCallback<T> | null): void;
}
export interface IWatchedValueReadonly<T> extends IObservableValueReadOnly<T> {
	/**
	 * Subscribe to watched value changes
	 * @param  {(value:T)=>void} callback - callback to be evoked when change occurs
	 * @param  {WatchedValueSubscribeOptions} [options] - watch subscriber options
	 */
	subscribe(callback: (value: T) => void, options?: WatchedValueSubscribeOptions): void;
	/**
	 * Unsubscribe to watched value changes
	 * @param  {((value:T)=>void)|null} [callback] - callback to remove
	 */
	unsubscribe(callback?: ((value: T) => void) | null): void;
	/**
	 * A simplified version of subscription, with promise-like interface, generally for using with boolean-valued watched values
	 * @param  {WatchedValueCallback<T>} callback - a function to be called when the value became `true`. `once` and `callWithLast` are implicitly set to true.
	 */
	when(callback: WatchedValueCallback<T>): void;
}
export interface IndividualPosition extends IndividualPositionBase, CustomFields {
}
/**
 * Describes an individual position.
 */
export interface IndividualPositionBase {
	/** Individual position ID. Usually id should be equal to brokerSymbol */
	id: string;
	/** Individual position open date (UNIX timestamp in milliseconds) */
	date: number;
	/** Symbol name */
	symbol: string;
	/** Individual position Quantity */
	qty: number;
	/** Individual position Side */
	side: Side;
	/** Individual position price */
	price: number;
}
export interface InstrumentInfo {
	/** Quantity field step and boundaries */
	qty: QuantityMetainfo;
	/** Value of 1 pip for the instrument in the account currency */
	pipValue: number;
	/** Size of 1 pip (e.g., 0.0001 for EURUSD) */
	pipSize: number;
	/** Minimal price change (e.g., 0.00001 for EURUSD). Used for price fields. */
	minTick: number;
	/** Lot size */
	lotSize?: number;
	/** Instrument type. `forex` enables negative pips. You can check that in Order Ticket. */
	type?: SymbolType;
	/** Units of quantity or amount. Displayed instead of the Units label in the Quantity/Amount field. */
	units?: string;
	/** Display name for the symbol */
	brokerSymbol?: string;
	/** A description to be displayed in the UI dialogs. */
	description: string;
	/** Number of decimal places of DOM asks/bids volume (optional, 0 by default). */
	domVolumePrecision?: number;
	/** Leverage */
	leverage?: string;
	/**
	 * The margin requirement for the instrument. A 3% margin rate should be represented as 0.03.
	 */
	marginRate?: number;
	/** Minimal price change for limit price field of the Limit and Stop Limit order. If set it will override the `minTick` value. */
	limitPriceStep?: number;
	/** Minimal price change for stop price field of the Stop and Stop Limit order. If set it will override the `minTick` value. */
	stopPriceStep?: number;
	/** Array of strings with valid duration values. You can check that in Order Ticket. */
	allowedDurations?: string[];
	/**
	 * Dynamic minimum price movement.
	 * It is used if the instrument's minimum price movement changes depending on the price range.
	 *
	 * For example: `0.01 10 0.02 25 0.05`, where `minTick` is `0.01` for a price less than `10`, `minTick` is `0.02` for a price less than `25`, `minTick` is `0.05` for a price more and equal than `25`.
	 */
	variableMinTick?: string;
	/** Instrument currency that is displayed in Order Ticket */
	currency?: string;
	/** The first currency quoted in a currency pair. Used for crypto currencies only. */
	baseCurrency?: string;
	/** The second currency quoted in a currency pair. Used for crypto currencies only. */
	quoteCurrency?: string;
	/** The value represented by a full point of price movement in the contract currency. This value is used to calculate the Total Value (symbol currency) of the order. */
	bigPointValue?: number;
	/** The value represents how much price is multiplied in relation to base monetary unit. */
	priceMagnifier?: number;
	/** Supported order types for the instrument */
	allowedOrderTypes?: OrderType[];
}
/** Show a custom message with the reason why the symbol cannot be traded */
export interface IsTradableResult {
	/**
	 * Is the symbol tradable
	 */
	tradable: boolean;
	/**
	 * Reason is displayed in Order Ticket
	 */
	reason?: string;
	/** Solution available to user to resolve the issue */
	solutions?: TradableSolutions;
	/** shortReason is displayed in the legend */
	shortReason?: string;
}
/**
 * An API object representing leverage info for an order.
 */
export interface LeverageInfo {
	/**
	 * The title for any Leverage Dialogs shown by the library.
	 */
	title: string;
	/**
	 * The leverage.
	 */
	leverage: number;
	/**
	 * The minimum leverage value.
	 */
	min: number;
	/**
	 * The maximum leverage value.
	 */
	max: number;
	/**
	 * The mimimum change between leverage values.
	 */
	step: number;
}
/**
 * An API object representing an order. Used when requesting leverage information from a broker.
 */
export interface LeverageInfoParams {
	/**
	 * The order symbol.
	 */
	symbol: string;
	/**
	 * The type of the order.
	 */
	orderType: OrderType;
	/**
	 * The order side. Buy or sell.
	 */
	side: Side;
	/**
	 * Custom data for the broker.
	 */
	customFields?: CustomInputFieldsValues;
}
/**
 * An API object representing some messages describing the leverage value set by the user.
 *
 * Shown in any Leverage Dialogs.
 */
export interface LeveragePreviewResult {
	/**
	 * Informative messages about the leverage value.
	 */
	infos?: string[];
	/**
	 * Warnings about the leverage value.
	 */
	warnings?: string[];
	/**
	 * Errors about the leverage value.
	 */
	errors?: string[];
}
/**
 * An API object representing an order and leverage. Used when requesting that a broker updates a order's leverage.
 */
export interface LeverageSetParams extends LeverageInfoParams {
	/**
	 * The requested leverage value.
	 */
	leverage: number;
}
/**
 * An API object representing a response containing the leverage value for a user.
 */
export interface LeverageSetResult {
	/**
	 * The leverage.
	 */
	leverage: number;
}
/** Separator for a dropdown or context menu */
export interface MenuSeparator extends ActionDescription {
	/** Is a menu separator */
	separator: boolean;
}
export interface NegativeBaseInputFieldValidatorResult extends BaseInputFieldValidatorResult {
	/** @inheritDoc */
	valid: false;
	/** Reason why base input value is invalid  */
	errorMessage: string;
}
/**
 * Interface for an URL which will be opened
 */
export interface OpenUrlSolution {
	/**
	 * Link to be opened
	 */
	openUrl: {
		/** URL to be opened */
		url: string;
		/** text for solution button */
		text: string;
	};
}
export interface OrderDialogOptions extends TradingDialogOptions {
	/**
	 * Using this flag, you can change `Trade Value` to `Total` in the *Order Info* section of the [Order Ticket](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/order-ticket.md).
	 */
	showTotal?: boolean;
}
export interface OrderDuration {
	/**
	 * type is OrderDurationMetaInfo.value
	 */
	type: string;
	/** Order duration time */
	datetime?: number;
}
/**
 * Order duration options that determine how long the order remains active.
 * Refer to [Set order duration](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/order-ticket.md#set-order-duration) for more information.
 */
export interface OrderDurationMetaInfo {
	/** If it is set to `true`, then the Display date control in Order Ticket for this duration type will be displayed. */
	hasDatePicker?: boolean;
	/** If it is set to `true`, then the Display time control in Order Ticket for this duration type will be displayed. */
	hasTimePicker?: boolean;
	/**
	 * Default duration.
	 * Only one duration object in the durations array can have a `true` value for this field.
	 * The default duration will be used when the user places orders in the silent mode and it will be the selected one when the user opens Order Ticket for the first time.
	 */
	default?: boolean;
	/** Localized title of the duration. The title will be displayed in the Duration control of Order Ticket. */
	name: string;
	/** Duration identifier */
	value: string;
	/** A list of order types for which this duration type will be displayed in the Duration control of Order Ticket. Default value is `[OrderType.Limit, OrderType.Stop, OrderType.StopLimit]`. */
	supportedOrderTypes?: OrderType[];
}
export interface OrderOrPositionMessage {
	/** Type of message about the order or position */
	type: OrderOrPositionMessageType;
	/** Message content */
	text: string;
}
/** Describes the result of the order preview. */
export interface OrderPreviewResult {
	/** Order preview section */
	sections: OrderPreviewSection[];
	/** Confirmation ID. A unique identifier that should be passed to `placeOrder` method */
	confirmId?: string;
	/** Warning messages */
	warnings?: string[];
	/** Error messages */
	errors?: string[];
}
/**
 * Describes a single order preview section.
 * Order preview can have multiple sections that are divided by separators and may have titles.
 */
export interface OrderPreviewSection {
	/** Order preview items. Each item is a row of the section table. */
	rows: OrderPreviewSectionRow[];
	/** Optional title of the section. */
	header?: string;
}
/**
 * Describes a single row of a section table of the order preview.
 */
export interface OrderPreviewSectionRow {
	/** Description of the item. */
	title: string;
	/** Formatted value of the item. */
	value: string;
}
export interface OrderRule {
	/** Order ID */
	id: string;
	/** Severity of Order Rule */
	severity: "warning" | "error";
}
/**
 * Input value of the Order Ticket.
 * This information is not sufficient to place an order.
 */
export interface OrderTemplate {
	/** Symbol identifier */
	symbol: string;
	/** Order Type */
	type?: OrderType;
	/** order / execution side */
	side?: Side;
	/** Order quantity */
	qty?: number;
	/** Type of Stop Order */
	stopType?: StopType;
	/** Order stop price */
	stopPrice?: number;
	/** Order limit price */
	limitPrice?: number;
	/** Order Take Profit (Brackets) */
	takeProfit?: number;
	/** Order Stop loss (Brackets) */
	stopLoss?: number;
	/** Order Trailing stop (Brackets) */
	trailingStopPips?: number;
	/** Duration or expiration of an order */
	duration?: OrderDuration;
	/** Custom input fields */
	customFields?: CustomInputFieldsValues;
}
export interface PipValues {
	/** value of 1 pip if you buy */
	buyPipValue: number;
	/** value of 1 pip if you sell */
	sellPipValue: number;
}
export interface PlaceOrderResult {
	/** Order id */
	orderId?: string;
}
export interface PlacedOrder extends PlacedOrderBase, CustomFields {
}
/**
 * An object that contains information about a placed order.
 */
export interface PlacedOrderBase {
	/** Order ID */
	id: string;
	/** Symbol name */
	symbol: string;
	/** Order type */
	type: OrderType;
	/** Order side (buy or sell) */
	side: Side;
	/** Order quantity (double) */
	qty: number;
	/** Order status */
	status: OrderStatus;
	/** Stop loss price (double). Available when Brackets are enabled */
	stopLoss?: number;
	/** Trailing stop Pips value (double). Available when Brackets are enabled */
	trailingStopPips?: number;
	/** Stop Loss type. It should be set to 1 (StopType.TrailingStop) for trailing stop orders. */
	stopType?: StopType;
	/** Take profit price (double). Available when Brackets are enabled */
	takeProfit?: number;
	/** Order duration */
	duration?: OrderDuration;
	/**
	 * An object that contains the results of broker specific user inputs (for example a digital signature).
	 * There are two possible kinds of custom fields: an input field with a checkbox and a custom combobox.
	 */
	customFields?: CustomInputFieldsValues;
	/** Filled order quantity (double) */
	filledQty?: number;
	/** Average fulfilled price for the order (double) */
	avgPrice?: number;
	/** Last update time (unix timestamp in milliseconds) */
	updateTime?: number;
	/** Price for the limit order (double) */
	limitPrice?: number;
	/** Price for the stop order (double) */
	stopPrice?: number;
	/** Message describing the state of the order */
	message?: OrderOrPositionMessage;
}
export interface Position extends PositionBase, CustomFields {
}
/**
 * Describes a single position.
 */
export interface PositionBase {
	/** Position ID. Usually id should be equal to brokerSymbol */
	id: string;
	/** Symbol name */
	symbol: string;
	/** Position Quantity (positive number) */
	qty: number;
	/** Short position quantity */
	shortQty?: number;
	/** Long position quantity */
	longQty?: number;
	/** Position Side */
	side: Side;
	/** Average price */
	avgPrice: number;
	/** Message describing the state of the position */
	message?: OrderOrPositionMessage;
}
export interface PositionDialogOptions extends TradingDialogOptions {
}
export interface PositiveBaseInputFieldValidatorResult extends BaseInputFieldValidatorResult {
	/** @inheritDoc */
	valid: true;
}
/**
 * Output value of the Order Ticket and input value of the broker's place order command.
 * This information is sufficient to place an order.
 */
export interface PreOrder extends OrderTemplate {
	/** @inheritDoc */
	symbol: string;
	/** @inheritDoc */
	type: OrderType;
	/** @inheritDoc */
	side: Side;
	/** @inheritDoc */
	qty: number;
	/** Current Quotes */
	currentQuotes?: AskBid;
	/**
	 * It is set to `true`, if the order closes a position.
	 */
	isClose?: boolean;
}
/**
 * Quantity field step and boundaries
 */
export interface QuantityMetainfo {
	/** Minimum quantity */
	min: number;
	/** Maximum quantity */
	max: number;
	/** Quantity step size */
	step: number;
	/** Quantity step size for scrolling */
	uiStep?: number;
	/** Default quantity value */
	default?: number;
}
export interface SingleBrokerMetaInfo {
	/**
	 * Broker Configuration Flags
	 */
	configFlags: BrokerConfigFlags;
	/**
	 * Optional field. You can use it if you have custom fields in orders or positions that should be taken into account when showing notifications.
	 *
	 * @example
	 * if you have field `additionalType` in orders and you want the chart to show a notification when it is changed, you should set:
	 * ```javascript
	 * customNotificationFields: ['additionalType']
	 * ```
	 */
	customNotificationFields?: string[];
	/**
	 * List of order duration options that determine how long the order remains active.
	 * Specifying `durations` enables a drop-down menu in the Order Ticket for supported orders.
	 * Refer to [Set order duration](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/order-ticket.md#set-order-duration) for more information.
	 *
	 * The objects have the following keys: `{ name, value, hasDatePicker?, hasTimePicker?, default?, supportedOrderTypes? }`.
	 */
	durations?: OrderDurationMetaInfo[];
	/**
	 * Order Rules
	 */
	orderRules?: OrderRule[];
	/**
	 * This optional field can be used to replace the standard Order Ticket and the Add Protection dialogs with your own.
	 * Values of the following two fields are functions that are called by the Trading Platform to show the dialogs. Each function shows a dialog and returns a `Promise` object that should be resolved when the operation is finished or cancelled.
	 *
	 * **NOTE:** The returned `Promise` object should be resolved with either `true` or `false` value.
	 *
	 * @example
	 * ```ts
	 * customUI: {
	 *     showOrderDialog?: (order: Order, focus?: OrderTicketFocusControl) => Promise<boolean>;
	 *     showPositionDialog?: (position: Position | IndividualPosition, brackets: Brackets, focus?: OrderTicketFocusControl) => Promise<boolean>;
	 *     showCancelOrderDialog?: (order: Order) => Promise<boolean>;
	 *     showClosePositionDialog?: (position: Position) => Promise<boolean>;
	 * }
	 * ```
	 */
	customUI?: BrokerCustomUI;
}
export interface SortingParameters {
	/** `property` of the data object that will be used for sorting */
	property: string;
	/** Ascending sorting order (default `true`) - If it is `false`, then initial sorting will be in descending order */
	asc?: boolean;
}
/**
 * The interface that describes the mapping of values in the {@link AccountManagerColumnBase.formatter} and {@link AccountManagerColumnBase.dataFields} properties of the Account Manager columns.
 * Refer to the [Value formatters](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/account-manager/value-formatters.md) section for more information.
 */
/* eslint-disable jsdoc/require-jsdoc */
export interface StandardFormattersDependenciesMapping {
	[StandardFormatterName.Default]: string[];
	[StandardFormatterName.Symbol]: [
		brokerSymbolProperty: string,
		symbolProperty: string,
		message: string
	] | [
		brokerSymbolProperty: string,
		symbolProperty: string
	];
	[StandardFormatterName.Side]: [
		sideProperty: string
	];
	[StandardFormatterName.PositionSide]: [
		sideProperty: string
	];
	[StandardFormatterName.Text]: string[];
	[StandardFormatterName.Type]: [
		orderTypeProperty: string,
		parentIdProperty: string,
		stopTypeProperty: string
	];
	[StandardFormatterName.FormatPrice]: [
		priceProperty: string
	];
	[StandardFormatterName.FormatPriceForexSup]: [
		priceProperty: string
	];
	[StandardFormatterName.Status]: [
		statusProperty: string
	];
	[StandardFormatterName.Date]: [
		dateProperty: string
	];
	[StandardFormatterName.LocalDate]: [
		dateProperty: string
	];
	[StandardFormatterName.DateOrDateTime]: [
		dateProperty: string
	];
	[StandardFormatterName.LocalDateOrDateTime]: [
		dateProperty: string
	];
	[StandardFormatterName.Fixed]: [
		valueProperty: string
	];
	[StandardFormatterName.FixedInCurrency]: [
		valueProperty: string,
		currencyProperty: string
	];
	[StandardFormatterName.VariablePrecision]: [
		valueProperty: string
	];
	[StandardFormatterName.Pips]: [
		pipsProperty: string
	];
	[StandardFormatterName.IntegerSeparated]: [
		valueProperty: string
	];
	[StandardFormatterName.FormatQuantity]: [
		qtyProperty: string
	];
	[StandardFormatterName.Profit]: [
		profitProperty: string
	];
	[StandardFormatterName.ProfitInInstrumentCurrency]: [
		profitProperty: string,
		currencyProperty: string
	];
	[StandardFormatterName.Percentage]: [
		valueProperty: string
	];
	[StandardFormatterName.MarginPercent]: [
		valueProperty: string
	];
	[StandardFormatterName.Empty]: [
	];
}
/* eslint-enable jsdoc/require-jsdoc */
export interface SuccessFormatterParseResult<T> extends FormatterParseResult {
	/** @inheritDoc */
	res: true;
	/** Returned value once parsing is done */
	value: T;
	/** Optional value returned by the default formatter */
	suggest?: string;
}
export interface SymbolSpecificTradingOptions {
	/** Array of strings with valid duration values. You can check that in Order Ticket. */
	allowedDurations?: string[];
	/** Supported order types for the instrument. */
	allowedOrderTypes?: OrderType[];
	/**
	 * Whether order brackets are supported for the symbol.
	 * Defaults to the value in the config.
	 */
	supportOrderBrackets?: boolean;
	/**
	 * Using this flag you can disable adding brackets to the existing order.
	 */
	supportAddBracketsToExistingOrder?: boolean;
	/**
	 * Using this flag you can disable existing order's brackets modification. If you set it to `false`,
	 * additional fields will be disabled in Order Ticket on the chart,
	 */
	supportModifyBrackets?: boolean;
	/**
	 * Whether position brackets are supported for the symbol.
	 * Defaults to the value in the config.
	 */
	supportPositionBrackets?: boolean;
	/**
	 * Whether trade brackets are supported for the symbol.
	 * Defaults to the value in the config.
	 */
	supportIndividualPositionBrackets?: boolean;
	/**
	 * Whether position reversing is supported for the symbol.
	 * Defaults to the value in the config.
	 */
	supportReversePosition?: boolean;
	/**
	 * A symbol-specific message that can be used to warn users.
	 */
	warningMessage?: string;
}
export interface TableFormatterInputs<T extends TableFormatterInputValues = TableFormatterInputValues> {
	/** Array of values to be formatted. Values are obtained by extracting dependent properties from the data object. */
	values: T extends [
		...args: infer A
	] ? [
		...A
	] : never;
	/** Optional field. It is array of previous values so you can compare and format accordingly. It exists if current column has the `highlightDiff: true` key. */
	prevValues?: Partial<T extends [
		...args: infer A
	] ? [
		...A
	] : never>;
}
export interface TextWithCheckboxFieldCustomInfo {
	/** Title for the checkbox */
	checkboxTitle: string;
	/**
	 * Using `asterix` property you can manage input type.
	 * If `asterix` is set to `true` then a password input will be rendered.
	 */
	asterix?: boolean;
}
export interface TextWithCheckboxFieldMetaInfo extends CustomInputFieldMetaInfo {
	/** @inheritDoc */
	inputType: "TextWithCheckBox";
	/** @inheritDoc */
	value: TextWithCheckboxValue;
	/** @inheritDoc */
	customInfo: TextWithCheckboxFieldCustomInfo;
	/** @inheritDoc */
	validator?: TextInputFieldValidator;
}
export interface TextWithCheckboxValue {
	/** Checkbox text */
	text: string;
	/** Whether the checkbox is checked */
	checked: boolean;
}
export interface TradeContext {
	/** Symbol name */
	symbol: string;
	/** Symbol display name */
	displaySymbol: string;
	/** Price value */
	value: number | null;
	/** Formatted value */
	formattedValue: string;
	/** Previous value */
	last: number;
}
export interface TradingDialogOptions {
	/** Custom fields that are displayed in the Order Ticket.
	 * Refer to the [Add custom fields](https://www.tradingview.com/charting-library-docs/latest/trading_terminal/order-ticket.md#add-custom-fields) section for more information.
	 *
	 * **Example**
	 * ```javascript
	 * customFields: [
	 *     {
	 *         inputType: 'TextWithCheckBox',
	 *         id: '2410',
	 *         title: 'Digital Signature',
	 *         placeHolder: 'Enter your personal digital signature',
	 *         value: {
	 *             text: '',
	 *             checked: false,
	 *         },
	 *         customInfo: {
	 *             asterix: true,
	 *             checkboxTitle: 'Save',
	 *         },
	 *     }
	 * ]
	 * ```
	 *
	 */
	customFields?: TradingDialogCustomField[];
}
export interface TradingQuotes {
	/** Trade */
	trade?: number;
	/** Quote size */
	size?: number;
	/** Bid price */
	bid?: number;
	/** Bid quantity */
	bid_size?: number;
	/** Ask price */
	ask?: number;
	/** Ask quantity */
	ask_size?: number;
	/** Price spread */
	spread?: number;
	/** Whether quotes are delayed */
	isDelayed?: boolean;
	/** Whether quotes are halted */
	isHalted?: boolean;
	/** Whether quotes are hard to borrow */
	isHardToBorrow?: boolean;
	/** Whether quotes are can not be shorted */
	isNotShortable?: boolean;
}
export interface WatchedValueSubscribeOptions {
	/** Subscribe for only one update, and remove subscription afterwards. (the callback will be executed only once) */
	once?: boolean;
	/** if it is set to true then the callback will be executed with the previous value (if available) */
	callWithLast?: boolean;
}
export type AccountId = Nominal<string, "AccountID">;
export type AccountManagerColumn = {
	[K in StandardFormatterName | FormatterName]: AccountManagerColumnBase<K>;
}[StandardFormatterName | FormatterName];
/**
 * Describes a single action to put it into a dropdown or a context menu.
 */
export type ActionMetaInfo = ActionDescriptionWithCallback | MenuSeparator;
/**
 * The Ask and Bid quotes.
 */
export type AskBid = Required<Pick<TradingQuotes, "ask" | "bid">>;
export type CellAlignment = "left" | "right";
export type CustomFieldPossibleTypes = "TextWithCheckBox" | "ComboBox" | "Checkbox";
/**
 * A function that takes an {@link TableFormatterInputs} object and returns a `string` or an `HTMLElement`.
 */
export type CustomTableFormatElementFunction<T extends TableFormatterInputValues = TableFormatterInputValues> = (inputs: TableFormatterInputs<T>) => undefined | string | HTMLElement;
export type FormatterName = Nominal<string, "FormatterName">;
/**
 * Input field validator
 * @param  {any} value - value to be validated
 */
export type InputFieldValidator = (value: any) => InputFieldValidatorResult;
export type InputFieldValidatorResult = PositiveBaseInputFieldValidatorResult | NegativeBaseInputFieldValidatorResult;
export type LanguageCode = "ar" | "zh" | "ca_ES" | "en" | "fr" | "de" | "he_IL" | "id_ID" | "it" | "ja" | "ko" | "pl" | "pt" | "ru" | "es" | "sv" | "th" | "tr" | "vi" | "ms_MY" | "zh_TW";
export type LeverageParams = LeverageInfoParams | LeverageSetParams;
export type Order = PlacedOrder | BracketOrder;
export type OrderTableColumn = AccountManagerColumn & {
	/**
	 * An optional numeric array of order statuses that is applied to order columns only. If it is available then the column will be displayed in the specified tabs of the status filter only.
	 *
	 * Here is the list of possible order statuses:
	 *
	 * 0 - All
	 * 1 - Canceled
	 * 2 - Filled
	 * 3 - Inactive
	 * 5 - Rejected,
	 * 6 - Working
	 */
	supportedStatusFilters?: OrderStatusFilter[];
};
export type SuggestedQtyChangedListener = (newQty: number) => void;
export type SymbolType = "stock" | "index" | "forex" | "futures" | "bitcoin" | "crypto" | "undefined" | "expression" | "spread" | "cfd" | "economic" | "equity" | "dr" | "bond" | "right" | "warrant" | "fund" | "structured" | "commodity" | "fundamental" | "spot" | "swap";
/**
 * A function that takes an {@link TableFormatterInputs} object and returns a `string`.
 */
export type TableFormatTextFunction<T extends TableFormatterInputValues = TableFormatterInputValues> = (inputs: TableFormatterInputs<T>) => string;
export type TableFormatterInputValue = any;
export type TableFormatterInputValues = TableFormatterInputValue[];
export type TextInputFieldValidator = (value: string) => InputFieldValidatorResult;
/**
 * `TradableSolutions` has one of the following keys:
 * - `changeAccount` - id of a sub-account suitable for trading the symbol
 * - `changeSymbol` - the symbol suitable for trading with current sub-account
 * - `openUrl` - the object with URL to be opened and text for solution button
 */
export type TradableSolutions = ChangeAccountSolution | ChangeSymbolSolution | OpenUrlSolution;
export type TradingDialogCustomField = CheckboxFieldMetaInfo | TextWithCheckboxFieldMetaInfo | CustomComboBoxMetaInfo;
export type WatchedValueCallback<T> = (value: T) => void;

export as namespace TradingView;

export {};
